<?php

//require __DIR__ . '/dep/vendor/autoload.php';

include_once(          "CSV.php" );
include_once(      "strings.php" );
include_once(      "jstream.php" );
include_once(          "log.php" );
include_once(          "ssl.php" );
include_once(        "mysql.php" );
include_once(        "input.php" );
include_once( "experimental.php" );

$args = isset( $argv ) ? $argv : array();

main( $args );

function main( $argv )
{
    if ( getenv( "USE_SYSLOG_PROGRAMNAME" ) ) define( "USE_SYSLOG_PROGRAMNAME", getenv( "USE_SYSLOG_PROGRAMNAME" ) );
    if ( getenv( "USE_SYSLOG_FACILITY"    ) ) define( "USE_SYSLOG_FACILITY",    getenv( "USE_SYSLOG_FACILITY"    ) );

    if ( defined( "USE_SYSLOG_PROGRAMNAME" ) && defined( "USE_SYSLOG_FACILITY" ) )
    {
        openlog( USE_SYSLOG_PROGRAMNAME, LOG_PID, NeoLog_DecodeFacility( USE_SYSLOG_FACILITY ) );
    }

    switch( $_SERVER['REQUEST_METHOD'] )
    {
    case "OPTIONS":
    //case "PUT":
    //case "DELETE":
        NeoLog( "Ignoring unsupported method " . $_SERVER['REQUEST_METHOD'] . ":" . $_SERVER["REDIRECT_URL"] . "?" . urldecode( file_get_contents( 'php://input') ) );
        break;

    default:
        $start = microtime( TRUE );
        $delta = 0;
        $use_buffered = ( "TRUE" == getenv( "USE_BUFFERED" ) );

        if
        (
            !string_has_prefix( $_SERVER["REDIRECT_URL"], "/auth/" )
            &&
            !string_has_prefix( $_SERVER["REDIRECT_URL"], "/api/"  )
            &&
            !string_has_prefix( $_SERVER["REDIRECT_URL"], "/raw/"  )
            &&
            !string_has_prefix( $_SERVER["REDIRECT_URL"], "/v"     )
        )
        {
            NeoLog( "Ignoring request:" . $_SERVER["REDIRECT_URL"] . "?" . urldecode( file_get_contents( 'php://input') ) );
            exit();
        }
        else
        {
            $method = sprintf( "%4s", $_SERVER['REQUEST_METHOD'] );

            if ( "TRUE" == getenv( "LOG_REQUEST_PARAMETERS" ) )
            {
                $request = RetrieveRequest();
            
                if ( array_key_exists( "password", $_REQUEST ) )
                {
                    $request["password"] = "[hidden]";
                }

                $encoded = http_build_query( $request );

                NeoLog( "Starting new request, $method " . $_SERVER["REDIRECT_URL"] . "?" . $encoded );
            }
            else
            {
                NeoLog( "Starting new request, $method " . $_SERVER["REDIRECT_URL"] );
            }
        }

        NeoLogIn();

        $environment = DetermineEnvironment         ( $argv                                                                         );
        $sp_name     = TranslateSPName              ( $environment["redirect_url"]                                                  );
                       CheckIfTunnelRequest         ( $sp_name,     $environment                                                    );
        $db_domain   = DetermineDatabaseLookupDomain( $environment["server_name"]                                                   );
        $db_info     = LookupDatabaseInfo           ( $db_domain                                                                    );
        $db_version  = LookupDatabaseVersion        ( $db_info                                                                      );
        //             CheckIfDownloadRequest       ( $db_info,     $db_version, $sp_name, $environment                             );
        $sp_call     = MapRequestToSPCall           ( $db_info,     $db_version, $sp_name, $environment["request"]                  );
        $ret         = CallStoredProcedure          ( $db_info,     $db_version, $sp_name, $sp_call, $environment                   );
                       SetSessionIDCookie           ( $sp_name,     $ret                                                            );
        $response    = CreateResponse               ( $environment, $ret, ($delta = ceil( (microtime( TRUE ) - $start) * 1000000 )) );
                       OutputResponse               ( $sp_name,     $response, $environment["request"], $use_buffered               );
        //             LogToPubSub                  ( $environment, $db_info, $db_version, $sp_call, $response, $delta              );

        NeoLogOut();

        if ( "TRUE" == getenv( "LOG_STORED_PROCEDURE_CALL" ) )
        {
            //$count = 0;

            if ( is_a( $ret, 'Results' ) )
            {
                LogResult( $sp_name, $ret, "returned" );
            }
            else
            if ( is_a( $ret, 'EmptyResults' ) )
            {
                LogResult( $sp_name, $ret, "returned" );
            }
            else
            if ( is_a( $ret, 'ErrorResults' ) )
            {
                switch ( $ret->sqlstate )
                {
                case "45000":
                    LogResult( $sp_name, $ret, "threw INFO: "    . $ret->errorMessage, LOG_INFO );
                    break;

                case "45001":
                    LogResult( $sp_name, $ret, "threw NOTICE: "  . $ret->errorMessage, LOG_NOTICE );
                    break;

                case "45002":
                    LogResult( $sp_name, $ret, "threw WARNING: " . $ret->errorMessage, LOG_WARNING );
                    break;

                case "45003":
                default:
                    LogResult( $sp_name, $ret, "threw ERROR: "   . $ret->errorMessage, LOG_ERR );
                    break;
                }
            }
            else
            {
                NeoLog( "Called: " . $sp_name . ": no return" );
            }
        }

        NeoLog( "Finishing:" . $_SERVER["REDIRECT_URL"] . " $delta microseconds" );
    }

    closelog();
}

function LogResult( $sp_name, $ret, $verb, $priority = LOG_INFO )
{
    $count    = $ret->numRows();
    $inc      = 4000;
    $len      = strlen( $ret->sqlQuery );
    $message  = "Called: " . $ret->sqlQuery . ": $verb " . $count . " rows";

    if ( $len < $inc )
    {
        NeoLog( $message, $priority );
    }
    else
    {
        NeoLog( "Called: " . $sp_name . ": $verb " . $count . " rows", $priority );

        for ( $i = 0; $i < $len; $i += $inc )
        {
            NeoLog( substr( $ret->sqlQuery, $i, $inc ), $priority );
        }
    }
}

function DetermineEnvironment( $argv )
{
    NeoLog( "Determining environment" );
    NeoLogIn();

    $environment = null;

    if ( "cli" != php_sapi_name() )
    {
        NeoLog( "Determining envionrment from Apache" );
        $environment = DetermineEnvironmentFromApache();
    }
    else
    {
        NeoLog( "Determining envionrment from command-line arguments" );
        $environment = DetermineEnvironmentFromArguments( $argv );
    }

    if ( ("" == $environment["server_name"]) || ("" == $environment["redirect_url"]) )
    {
        NeoLog( "Invalid environment, no SERVER_NAME or REDIRECT_URL." );
        NeoLog( "Commandline usage:" );
        NeoLog( "php index.php --server-name api.example.com --redirect-url /api/users/example/ --request '{\"param1\":\"value\",...}'" );
        exit;
    }

    if ( !defined( "DB_USERNAME" ) ) define( "DB_USERNAME", "public" );
    if ( !defined( "DB_PASSWORD" ) ) define( "DB_PASSWORD", "public" );
    if ( !defined( "SSL_BASE"    ) ) define( "SSL_BASE", "/etc/mysql/ssl/" );

    NeoLog( "Configuration: username: " . DB_USERNAME . "; password: " . DB_PASSWORD . "; ssl_base: " . SSL_BASE );

    //
    //  Setup use of CSRF Token
    //

    if ( "" != array_get( $environment['request'], "wab_csrf_token" ) )
    {
        $environment['request']["sid"]
        =
        substr( array_get( $environment['request'], "sid"            ), 0, 32 )
        .
        substr( array_get( $environment['request'], "wab_csrf_token" ), 0, 32 );
    }

    NeoLogOut();

    return $environment;
}

function DetermineEnvironmentFromApache()
{
    NeoLogIn();

    $input        = urldecode( file_get_contents( 'php://input' ) );
    $request      = RetrieveRequest();
    $server_name  = "";
    $redirect_url = "";
    $requested    = "";

    if ( isset( $_SERVER ) && array_key_exists( "SERVER_NAME", $_SERVER ) )
    {
        $server_name = $_SERVER["SERVER_NAME"];
    }

    if ( isset( $_SERVER ) && array_key_exists( "REDIRECT_URL", $_SERVER ) )
    {
        $redirect_url = CanonicalisePath( $_SERVER["REDIRECT_URL"] );
    }

    NeoLog( "Checking headers for API Key" );
    NeoLogIn();
    foreach( apache_request_headers() as $header => $value )
    {
        $_value = Input::Filter( $value );

        if ( "Accept" == $header && !array_key_exists( "accept", $request ) )
        {
            $request["accept"]         = $_value;
            NeoLog( "request['accept'] = $_value" );
        }
        else
        if ( "Authorization" == $header && !array_key_exists( "authorization", $request ) )
        {
            $request["authorization"]         = $_value;
            NeoLog( "request['authorization'] = $_value" );

            if ( !array_key_exists( "sid", $request ) )
            {
                $request["sid"]         = trim( str_replace( 'Bearer', '', $_value ) );
                NeoLog( "request['sid'] = $_value (Bearer)" );
            }
        }
        else
        if ( "X-Access-ID" == $header && !array_key_exists( "accessid", $request ) )
        {
            $request["accessid"]         = $_value;
            NeoLog( "request['accessid'] = $_value" );
        }
        else
        if ( "X-Api-Key" == $header && !array_key_exists( "apikey", $request ) )
        {
            $request["apikey"]         = $_value;
            NeoLog( "request['apikey'] = $_value" );
        }
        else
        if ( "X-Auth-Token" == $header && !array_key_exists( "auth_token", $request ) )
        {
            $request["auth_token"]         = $_value;
            NeoLog( "request['auth_token'] = $_value" );
        }
        else
        if ( "X-CSRF-Token" == $header && !array_key_exists( "wab_csrf_token", $request ) )
        {
            $request["wab_csrf_token"]         = $_value;
            NeoLog( "request['wab_csrf_token'] = $_value" );
        }
        else
        if ( "X-Request-ID" == $header && !array_key_exists( "requestid", $request ) )
        {
            $request["requestid"]         = $_value;
            NeoLog( "request['requestid'] = $_value" );
        }
        else
        if ( "X-Session-ID" == $header && !array_key_exists( "sid", $request ) )
        {
            $request["sid"]         = $_value;
            NeoLog( "request['sid'] = $_value" );
        }
        else
        {
            NeoLog( "Header: $header: $_value" );
        }
    }
    NeoLogOut();
    NeoLogOut();

    if ( isset( $request ) )
    {
        NeoLog( "Request is set" );
        $request = string_has_prefix( $redirect_url, "/raw/" ) ? $request : Input::FilterInput( $request, null );
    }

    return array
    (
        "server_name"  => $server_name,
        "redirect_url" => $redirect_url,
        "request"      => $request
    );
}

function CanonicalisePath( $constant )
{
    if ( '.php' == substr( $constant, -4 ) )
    {
        $constant = dirname( $constant );
    }

    if ( '/' != substr( $constant, -1 ) )
    {
        $constant .= "/";
    }

    return $constant;
}

function DetermineEnvironmentFromArguments( $argv )
{
    $server_name  = "";
    $redirect_url = "";
    $request      = "";

    $next_is_server_name  = 0;
    $next_is_redirect_url = 0;
    $next_is_request      = 0;
    foreach( $argv as $arg )
    {
        if ( $next_is_server_name )
        {
            $server_name         = $arg;
            $next_is_server_name = 0;
        }
        else
        if ( $next_is_redirect_url )
        {
            $redirect_url         = $arg;
            $next_is_redirect_url = 0;
        }
        else
        if ( $next_is_request )
        {
            $request         = json_decode( $arg, TRUE );
            $next_is_request = 0;
        }
        else
        {
            switch ( $arg )
            {
            case "--server-name":
                $next_is_server_name = 1;
                break;

            case "--redirect-url":
                $next_is_redirect_url = 1;
                break;

            case "--request":
                $next_is_request = 1;
                break;
            }
        }
    }

    if ( !$request ) $request = array();

    return array
    (
        "server_name"  => $server_name,
        "redirect_url" => $redirect_url,
        "request"      => $request
    );
}

function RetrieveRequest()
{
    $get     = array_slice( $_REQUEST, 0, count( $_REQUEST ) );
    $payload = array();

    if ( "GET" != $_SERVER["REQUEST_METHOD"] )
    {
        $input = file_get_contents( 'php://input' );

        switch( DetermineContentType() )
        {
        case "application/json":
            if ( "{" == substr( $input, 0, 1 ) )
            {
                $object = json_decode( $input );

                if ( is_object( $object ) )
                {
                    if ( getenv( "PARSE_JSON" ) )
                    {
                        $payload = (array) $object;
                    }
                }
                else
                {
                    $get["json"] = $input;
                }
            }
            break;
        
        default:
            parse_str( $input, $payload );
        }
    }

    return array_merge( $get, $payload );
}

function DetermineContentType()
{
    $content_type = "";

    foreach( apache_request_headers() as $header => $value )
    {
        $_value = Input::Filter( $value );

        if ( "Content-Type" == $header )
        {
            $content_type = $value;
            break;
        }
    }

    return $content_type;
}

function TranslateSPName( $redirect_url )
{
    NeoLog( "Translating stored procedure name from url" );

    $bits = explode( '/', str_replace( '-', '_', $redirect_url ) );

    foreach( $bits as $bit )
    {
        if ( "" != trim( $bit ) && !ctype_alnum( str_replace( '_', '', $bit ) ) )
        {
            error_log( "Exiting, malicious request detected: $redirect_url" );
            http_response_code( 400 );
            flush();
            exit();
        }
    }

    if ( "" == $bits[0] ) $bits = array_slice( $bits, 1 );

    switch ( $bits[0] )
    {
    case 'api':
        $sp_name = trim( implode( '_', array_slice( $bits, 1 ) ), '_' );
        break;

    case 'auth':
        $sp_name = trim( implode( '_', array_slice( $bits, 0 ) ), '_' );
        break;

    case 'download':
        $sp_name = 'download';
        break;

    case 'raw':
        $sp_name = trim( implode( '_', array_slice( $bits, 0 ) ), '_' );
        break;

    case 'tunnel':
        $sp_name = 'tunnel';
        break;

    default:
        if ( "v" == substr( $bits[0], 0, 1 ) )
        {
            $version = substr( $bits[0], 1 );

            if ( !is_numeric( $version ) )
            {
                http_response_code( 402 );
                exit;
            }
            else
            {
                $sp_name = trim( implode( '_', array_slice( $bits, 0 ) ), '_' );
            }
        }
        else
        {
            http_response_code( 402 );
            exit;
        }
    }

    return $sp_name;
}

function CheckIfTunnelRequest( $sp_name, $environment )
{
    if ( "tunnel" != $sp_name )
    {
        //
        //  This is mysteriously not being executed...
        //

        NeoLog( "Checking if tunnel request - NO" );
    }
    else
    {
        NeoLog( "Checking if tunnel request - YES" );
        NeoLogIn();
        Tunnel( $environment );
        NeoLogOut();
    }
}

function Tunnel( $environment )
{
    $request       = $environment['request'];
    $method        = array_get( $request, "wab_method"        );
    $authorization = array_get( $request, "wab_authorization" );
    $host          = array_get( $request, "wab_host"          );
    $endpoint      = array_get( $request, "wab_endpoint"      );
    $accept        = array_get( $request, "wab_accept"        );
    $content_type  = array_get( $request, "wab_content_type"  );
    $content64     = array_get( $request, "wab_content64"     );
    $headers       = array_get( $request, "wab_headers"       );

    NeoLog( "WAB tunnel debug: $host$endpoint" );

    if ( ! ($method && $host && $endpoint && $accept && $content_type) )
    {
        header( "Content-Type: text/plain" );

        echo "Error: incomplete headers"         . "\n";
        echo "+ wab_method:       $method"       . "\n";
        echo "+ wab_host:         $host"         . "\n";
        echo "+ wab_endpoint:     $endpoint"     . "\n";
        echo "+ wab_accept:       $accept"       . "\n";
        echo "+ wab_content_type: $content_type" . "\n";
        echo "+ wab_content64:    $content64"    . "\n";
    }
    else
    {
        $curl         = SetupCURL( $method, $authorization, $host, $endpoint, $content_type, $content64, $accept, $headers );
        $response     = curl_exec ( $curl );
        $err          = curl_error( $curl );

        if ( $err )
        {
            header( "Content-Type: text/plain" );
            echo $err;
        }
        else
        {
            header( "Content-Type: $accept" );
            echo $response;
        }
    }

    ob_flush();
    flush();
    exit;
}

function SetupCURL( $method, $authorization, $host, $endpoint, $content_type, $content64, $accept, $headers )
{
    $curl = curl_init();

    $verify_host = 0;
    $verify_peer = 0;
    $verify = ! string_has_suffix( $host, ".local" );

    if ( $verify )
    {
        $verify_host = 1;
        $verify_peer = 1;
    }

    $headers = array
    (
        "Accept: $accept",
        "Content-Type: $content_type",
        "Cache-Control: no-cache"
    );

    if ( "" != $authorization )
    {
        $headers[] = "Authorization: $authorization";
    }

    $url      = $host . $endpoint;
    $content  = base64_decode( $content64 );

    if ( "GET" == $method )
    {
        $url     = $url . "?" . $content;
        $content = "";
    }
    else
    {
        $payload    = array();
        $parameters = explode( "&", $content );
        foreach ( $parameters as $keyval )
        {
            $bits = explode( "=", $keyval );
            $key  = $bits[0];
            $val  = $bits[1];

            $payload[$key] = $val;
        }
        $content = http_build_query( $payload );
    }

    error_log( "WAB Tunnel: $url" );

    curl_setopt_array(
        $curl,
        array
        (
            CURLOPT_URL            => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING       => "",
            CURLOPT_MAXREDIRS      => 10,
            CURLOPT_TIMEOUT        => 30,
            CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST  => $method,
            CURLOPT_POSTFIELDS     => $content,
            CURLOPT_HTTPHEADER     => $headers,
            CURLOPT_SSL_VERIFYHOST => 0,
            CURLOPT_SSL_VERIFYPEER => 0
        )
    );

    return $curl;
}

function DetermineDatabaseLookupDomain( $server_name )
{
    if ( string_has_suffix( $server_name, ".local" ) )
    {
        $db_domain = "local-db." . str_replace( '.local', '', $server_name );
    }
    else
    {
        $db_domain = "db." . $server_name;
    }

    NeoLog( "Determining database lookup domain name: $db_domain" ); 

    return $db_domain;
}

function LookupDatabaseInfo( $db_domain )
{
    $db_info = null;
    $db_temp = (object) array( "dbname" => null, "hosts" => null );

    if ( ($database = getenv( "USE_DATABASE" )) )
    {
        $db_info = ExtractDBInfoFromEnvText( $database );
    }
    else
    {
        if ( function_exists( "apcu_enabled" ) && apcu_enabled() && apcu_exists( $db_domain ) )
        {
            $db_cached = apcu_fetch( $db_domain );
            $db_name   = $db_cached->dbname;
            $db_host   = $db_cached->hosts[0];

            if ( $db_name && $db_host )
            {
                NeoLog( "Looking up database info: found cached: $db_name@$db_host" );
                $db_info = $db_cached;
            }
        }

        if ( ! $db_info )
        {
            try
            {
                $records = dns_get_record( $db_domain, DNS_TXT );
                $n       = (null != $records) ? count( $records ) : 0;

                if ( 0 < $n )
                {
                    $db_dns = ExtractDBInfoFromDNSRecords( $records );
                }
                else
                {
                    $db_dns = ExtractDBInfoFromDBDomain( $db_domain );
                }

                $db_name = $db_dns->dbname;
                $db_host = $db_dns->hosts[0];

                if ( $db_name && $db_host )
                {
                    NeoLog( "Looking up database info: DNS returned $n records: $db_name@$db_host" );
                    $db_info = $db_dns;
                }

                if ( $db_info && function_exists( "apcu_enabled" ) && apcu_enabled() )
                {
                    $ttl = 60; // seconds

                    apcu_add( $db_domain, $db_info, $ttl );
                    NeoLogIn();
                    NeoLog( "Caching: added to cache: $db_domain" );
                    NeoLogOut();
                }
            }
            catch ( Exception $ex )
            {
                NeoLog( "Exception while retrieving from DNS" );
                exit;
            }
        }
    }

    NeoLogIn();
    if ( ! $db_info )
    {
        NeoLogError( "503 -- error, could not determine database information, exiting." );
        http_response_code( 503 );
        exit;
    }
    else
    {
        //  Following will abort if cannot retrieve database connection.
        CheckDatabaseConnection( $db_info );
    }
    NeoLogOut();

    return $db_info;
}

function ExtractDBInfoFromDNSRecords( $records )
{
    foreach ( $records as $record )
    {
        $host    = array_key_exists(    "host", $record ) ? $record[   "host"] : "";
        $class   = array_key_exists(   "class", $record ) ? $record[  "class"] : "";
        $ttl     = array_key_exists(     "ttl", $record ) ? $record[    "ttl"] : "";
        $type    = array_key_exists(    "type", $record ) ? $record[   "type"] : "";
        $txt     = array_key_exists(     "txt", $record ) ? $record[    "txt"] : "";

        if ( $txt )
        {
            return ExtractDBInfoFromDNSText( $txt );
        }
    }

    return null;
}

function ExtractDBInfoFromDNSText( $txt )
{
    $bits = explode( "@", $txt );
        
    switch ( count( $bits ) )
    {
    case 2:
        $dbname = $bits[0];
        $hosts  = explode( ",", $bits[1] );
        break;
    
    case 1:
        $hosts  = explode( ",", $bits[0] );
        break;
    }

    return (object) array( "dbname" => $dbname, "hosts" => $hosts );
}

/*
 *  Important!!! Below code is different to similar code above.
 *  This function is called when the USE_DATABASE environment variable is used.
 *  This may either be: <database>,             e.g., base
 *  or                  <database>@[host1],..., e.g., base@db1.example.com,db2.example.com
 *
 *  In contrast, code above allows a DNS text record to have only hostnames.
 */
function ExtractDBInfoFromEnvText( $txt )
{
    $bits = explode( "@", $txt );
        
    switch ( count( $bits ) )
    {
    case 2:
        $dbname = $bits[0];
        $hosts  = explode( ",", $bits[1] );
        break;
    
    case 1:
        $dbname = $bits[0];
        $hosts  = array( "localhost" );
        break;
    }

    return (object) array( "dbname" => $dbname, "hosts" => $hosts );
}

function ExtractDBInfoFromDBDomain( $db_domain )
{
    $truncated = str_replace(    "www.", "", $db_domain );
    $truncated = str_replace( "ro-api-", "", $truncated );
    $truncated = str_replace( "ro-api.", "", $truncated );
    $truncated = str_replace(    "api-", "", $truncated );
    $truncated = str_replace(    "api.", "", $truncated );
    $bits      = explode( ".", $truncated );

    $n = count( $bits );

    for ( $i = 1; $i < $n; $i++ )
    {
        if ( ! IsTLD( $bits[$i], $i ) )
        {
            $dbname = $bits[$i];
            break;
        }
    }

    return (object) array( "dbname" => $dbname, "hosts" => array("localhost") );
}

function IsTLD( $domain_part, $level = 1 )
{
	switch ( $level )
	{
	case "1":
		switch ( $domain_part )
		{
		case "au":
		case "com":
		case "net";
		case "org";
		case "co":
		case "info":
		case "xyz":
		case "local":
			return true;
			
		default:
			return false;
		}
		break;

	case "2":
		switch ( $domain_part )
		{
		case "com":
		case "net";
		case "org";
		case "id":
		case "conf":
			return true;
			
		default:
			return false;
		}
		break;
	
	default:
		return false;
	}
}

function CheckDatabaseConnection( $db_info )
{
    if ( !defined( "DB_HOSTNAME" ) ) define( "DB_HOSTNAME", $db_info->hosts[0] );

    NeoLog( "Checking database connection to: " . $db_info->hosts[0] );
    NeoLogIn();

    $m = MySQLCreate();

    NeoLogOut();

    return TRUE;
}

function LookupDatabaseVersion( $db_info )
{
	$db_version = "";
    $db_name    = "";
    $db_hosts   = implode( ', ', $db_info->hosts );
    $db_host    = $db_info->hosts[0];

    $mysql  = MySQLCreate();
    $tuples = MySQLInfo( $mysql, "SHOW DATABASES" );

    foreach( $tuples as $tuple )
    {
        $tmp = $tuple["Database"];

        if ( string_has_prefix( $tmp, $db_info->dbname ) )
        {
            $db_name = $tmp;
        }
    }
    $db_version = $db_name ? str_replace( $db_info->dbname, "", $db_name ) : $db_version;

    if ( $db_name )
    {
        NeoLog( "Determining target database version: found: $db_name" );
    }
    else
    {
        NeoLogError( "503 -- could not find database for: " . $db_info->dbname );
        http_response_code( 503 );
        exit;
    }

    return $db_version;
}

function CheckIfDownloadRequest( $db_info, $db_version, $sp_name, $environment )
{
    NeoLog( "Checking if tunnel request" );

    if ( "download" != $sp_name )
    {
        //
        //  This is mysteriously not being executed...
        //

        NeoLog( "Checking if download request - NO" );
    }
    else
    {
        NeoLog( "Performing download" );

        NeoLogIn();
        Download( $db_info, $db_version, $environment );
        NeoLogOut();
    }
}

function Download( $db_info, $db_version, $environment )
{
    $request = $environment['request'];
    $db      = $db_info->dbname . $db_version;
    $sid     = array_get( $request, "sid"   );
    $token   = array_get( $request, "token" );

    if ( $token )
    {
        $file = Retrieve( $db, $sid, $token );

        if ( $file )
        {
            $name    = $file->filename;
            $type    = $file->filetype;
            $size    = $file->filesize;
            $base64  = $file->base64;
            $content = base64_decode( $base64 );

            header( "Content-Type: application/octet-stream"                    );
            header( "Content-Length: $size"                                     );
            header( "Content-Disposition: attachment; filename=\"$name\""       );
            header( "Pragma: public"                                            );
            header( "Cache-Control: must-revalidate, post-check=0, pre-check=0" );
            echo $content;
        }
        else
        {
            header( "Content-Type: text/plain" );
            echo "File not found";
        }
    }
    else
    {
        header( "Content-Type: text/plain" );
        echo "No file specified";
    }

    ob_flush();
    flush();
    exit;
}

function Retrieve( $db, $sid, $token )
{
    $sp_call = (object) array
    (
        "proc_name" => "Base_Files_Retrieve_By_Token",
        "arguments" => array( $sid, $token )
    );

    $ret = MySQLProcedure( $db, $sp_call, $procedure );

    //NeoLog( "Retrieve: CALL $procedure" );

    return array_key_exists( 0, $ret ) ? $ret[0] : null;
}

function MapRequestToSPCall( $db_info, $db_version, $sp_name, $request )
{
    $sp_call   = null;
    $proc_name = null;
    $arguments = array();

    if ( "multiselect" != $sp_name )
    {
        NeoLog( "Determine Stored Procedure Call" );
        NeoLogIn();

        $db = $db_info->dbname . $db_version;

        //
        //  Create list of provisional sp names.
        //

        $provisional = array();

        switch( $_SERVER['REQUEST_METHOD'] )
        {
        case "GET":
        case "POST":
        case "PATCH":
        case "PUT":
        case "DELETE":
            $provisional[] = $sp_name . "_" . $_SERVER['REQUEST_METHOD'];
            break;
        }

        $provisional[] = $sp_name;
        if ( string_has_suffix( $sp_name, "_csv" ) )
        {
            $provisional[] = preg_replace( "/_csv\z/", "", $sp_name );
        }
        $provisional[] = $provisional[count($provisional)-1] . "_retrieve";

        //
        //  Use first procedure found that matches provisional name.
        //

        $result = null;
        foreach( $provisional as $prov )
        {
            $sql = "RetrieveParametersFor( '$db', '$prov' )";

            $result = MySQLFunction( $db, $sql );

            if ( "DEACTIVATED" == $result )
            {
                NeoLogError( "Database ($db) is deactivated." );
                http_response_code( 503 );
                exit( -1 );
            }
            else
            if ( null !== $result )
            {
                $proc_name = $prov;
                break;
            }
        }

        if ( !$proc_name )
        {
            NeoLog( "Could not find procedure for: " . $sp_name );
            http_response_code( 404 );
            return null;
        }
        else
        {
            $parameters = ConvertParametersToDictionary( $result );

            foreach ( $parameters as $parameter )
            {
                // Strip of leading '$' and make lowercase.
                $parameter_name = str_replace( '$', '', $parameter );
                $value          = null;

                if ( "wab_remote_address" == $parameter_name )
                {
                    $value = $_SERVER["REMOTE_ADDR"];
                }
                else
                if ( "sid" == strtolower( $parameter_name ) )
                {
                    $sid         = array_get( $request, "sid" );
                    $remote_addr = GetRemoteAddress();

                    if ( Experimental() )
                    {
                        $value = $sid . "@" . $remote_addr;
                    }
                    else
                    {
                        $value = $sid;
                    }
                }
                else
                if ( "apikey" == strtolower( $parameter_name ) )
                {
                    $value = array_get( $request, "apikey" ) . "@" . $_SERVER["REMOTE_ADDR"];
                }
                else
                if ( array_key_exists( $parameter_name, $request ) )
                {
                    $value = array_get( $request, $parameter_name );
                }
                else
                if ( array_key_exists( strtolower( $parameter_name ), $request ) )
                {
                    $value = array_get( $request, strtolower( $parameter_name ) );
                }
                else
                {
                    if ( ("PATCH" == $_SERVER["REQUEST_METHOD"]) )
                    {
                        $value = null;
                    }
                    else
                    {
                        $value = "";
                    }
                }

                $arguments[] = $value;
            }
        }

        NeoLogOut();
    }

    return (object) array( "proc_name" => $proc_name, "arguments" => $arguments );
}

function GetRemoteAddress()
{
	if ( array_key_exists( "X-Forwarded-For", $_SERVER ) && $_SERVER["X-Forwarded-For"] )
	{
		return strtok( $_SERVER["X-Forwarded-For"], "," );
	}
	else
	{
		return $_SERVER["REMOTE_ADDR"];
	}
}

function ConvertParametersToDictionary( $string )
{
    $parameters = array();
    $bits       = explode( ",", trim( $string ) );
    
    foreach ( $bits as $parameter )
    {
        $parts = explode( " ", trim( $parameter ) );
        
        if ( '_' != $parts[0][0] )
        {
            $parameters[] = $parts[0];

            if ( defined( "VERBOSE" ) && VERBOSE ) error_log( $parts[0] . " " . $parts[1] );
        }
    }
    
    return $parameters;
}

function CallStoredProcedure( $db_info, $db_version, $sp_name, $sp_call, $environment )
{
    $ret = null;

    if ( $sp_call )
    {
        NeoLog( "Call Stored Procedure" );
        NeoLogIn();

        $db = $db_info->dbname . $db_version;

        $cache_seconds = intval( getenv( "USE_CACHE_RESULTS_FOR_SECS" ) );

        if ( "multiselect" == $sp_name )
        {
            $ret = Multiselect   ( $db_info, $db_version, $environment );
        }
        else
        if ( $cache_seconds && function_exists( "apcu_enabled" ) && apcu_enabled() )
        {
            $key = $sp_call->proc_name . "(" . join( ",", $sp_call->arguments ) . ")";

            NeoLog( "Key: $key" );

            if ( apcu_exists( $key ) )
            {
                error_log( "Returning cached error value (cached for $cache_seconds seconds)" );
                $ret = apcu_fetch( $key );
            }
            else
            {
                $ret = MySQLProcedureResults( $db, $sp_call );

                if ( $ret instanceof ErrorResults )
                {
                    apcu_store( $key, $ret, $cache_seconds );
                }
            }
        }
        else
        {
            $ret = MySQLProcedureResults( $db, $sp_call );
        }

        NeoLogOut();
    }

    return $ret;
}

function Multiselect( $db_info, $db_version, $environment )
{
    $use_new        = true;

    $set            = array();
    $db             = $db_info->dbname . $db_version;
    $kinds          = explode( ',', array_get( $environment['request'], "kinds" ) );
    $sid            = array_get( $environment['request'], "sid"    );
    $filter         = array_get( $environment['request'], "filter" );

    #
    #   Create pseudo-request for select call
    #

    $entity_encoded      = array_get( $environment['request'], "json"   );
    $json                = html_entity_decode( $entity_encoded );
    $request             = (array) json_decode( $json );
    $request['filter']   = $filter;
    $request['sid']      = array_get( $environment['request'], "sid"    );

    foreach ( $kinds as $kind )
    {
        $sp_call = null;

        $bits = explode( ":", $kind );
        switch ( count( $bits ) )
        {
        case 1:
            $id          = "";
            $select_name = $bits[0];
            break;

        case 2:
            $id          = explode( ':', $kind )[0];
            $select_name = explode( ':', $kind )[1];
            break;

        default:
            break;
        }

        if ( $select_name )
        {
            $proc_name = "selects_$select_name";

            if ( $use_new )
            {
                $request['id'] = $id;
                $sp_call       = MapRequestToSPCall( $db_info, $db_version, $proc_name, $request );
                $tuples        = MySQLProcedure( $db, $sp_call, $procedure );
            }
            else
            {
                $arguments   = array();
                $arguments[] = $sid;
                $arguments[] = $id;
                $arguments[] = "";
                $arguments[] = $filter;
            
                $sp_call = (object) array
                (
                    "proc_name" => $proc_name,
                    "arguments" => $arguments
                );

                $tuples = MySQLProcedure( $db, $sp_call, $procedure );
            }

            if ( is_array( $tuples ) )
            {
                $set[] = (object) array( "id" => $id, "name" => $kind, "tuples" => $tuples, "call" => $procedure );
            }
            else
            {
                $set[] = (object) array( "id" => $id, "name" => $kind,  "error" => $tuples, "call" => $procedure );
            }
        }
    }

    return $set;
}

function SetSessionIDCookie( $sp_name, $ret )
{
    NeoLog( "Existing sessionid: " . array_get( $_COOKIE, "sid" ) );

    NeoLog( "USE_SESSION_COOKIE: " . getenv( "USE_SESSION_COOKIE" ) );

    if ( "FALSE" != getenv( "USE_SESSION_COOKIE" ) )
    {
        $is_local = ("8443" == $_SERVER['SERVER_PORT']);

        if ( !$is_local ) $is_local = string_has_suffix( $_SERVER['SERVER_NAME'], '.test' );

        if ( "auth_login" == $sp_name )
        {
            if ( is_a( $ret, 'Results' ) )
            {
                $tuple     = $ret->getFirst(); // Retrieve first tuple.
                $sessionid = $tuple->sessionid;
                $cookie    = "Set-Cookie: sid=";
                $cookie    = $cookie . $sessionid;

                if ( $is_local )
                {
                    // SameSite is causing issues with local API server
                    // that uses self-signed certificates.

                    $cookie = $cookie . "; path=/; HttpOnly;secure";
                }
                else
                {
                    $cookie = $cookie . "; path=/; HttpOnly;secure;SameSite=strict";
                }
        
                header( $cookie );

                NeoLog( "Setting Session ID Cookie: " . $sessionid );
            }
        }
        else
        if ( "auth_logout" == $sp_name )
        {
            header( "Set-Cookie: sid=deleted; path=/; HttpOnly;secure;SameSite=strict; expires=Thu, 01 Jan 1970 00:00:00 GMT" );
     
            NeoLog( "Clearing Session ID Cookie" );
        }
    }

    if ( "TRUE" == getenv( "USE_OAUTH2_TOKEN" ) )
    {
        if ( "auth_oauth2_token" == $sp_name )
        {
            if ( is_a( $ret, 'Results' ) )
            {
                $tuple = $ret->getFirst(); // Retrieve first tuple.

                if ( $tuple && $tuple->access_token )
                {
                    $access_token = $tuple->access_token;

                    echo '{' . '"access_token"' . ':' . '"' . $access_token . '"' . '}';

                    NeoLog( "Returning OAuth Token: " . $access_token );

                    exit( 0 );
                }
            }
        }
    }
} 

function CreateResponse( $environment, $ret, $microseconds)
{
    $request = $environment["request"];

    $response              = array();
    $response['URL'      ] = $environment["redirect_url"];
    $response['error'    ] = is_array( $ret ) || is_a( $ret, 'Results' ) || is_a( $ret, 'EmptyResults' ) ? "" : ( is_a( $ret, 'ErrorResults' ) ? $ret->errorMessage : $ret );
    $response['severity' ] = "";
    $response['failover' ] = "FALSE";
    $response['hostname' ] = "";
    $response['message'  ] = "";
    $response['status'   ] = is_array( $ret ) || is_a( $ret, 'Results' ) || is_a( $ret, 'EmptyResults' ) ? "OK" : "ERROR";
    $response['warning'  ] = "";
    $response['target_id'] = array_get( $request, "target_id" );
    $response['submit'   ] = array_get( $request, "submit"    );
    $response['limit'    ] = array_get( $request, "limit"     );
    $response['offset'   ] = array_get( $request, "offset"    );
    $response['microsecs'] = $microseconds;
    $response['results'  ] = is_array( $ret ) || is_a( $ret, 'Results' ) ? $ret : (is_a( $ret, 'EmptyResults' ) ? array() : null);

    if ( "TRUE" == getenv( "USE_HTTP_ERROR_CODES" ) )
    {
        if ( string_contains( $response["status"], "ERROR" ) )
        {
            $error_code = intval( substr( $response["error"], 0, 3 ) );
            
            if ( 0 != $error_code )
            {
                http_response_code( $error_code );

                $response["error"] = substr( $response["error"], 3 );
            }
            else
            if ( string_contains( $response["error"], "INVALID_APIKEY" ) )
            {
                http_response_code( 403 );
            }
            else
            if ( string_contains( $response["error"], "INVALID_AUTHORISATION" ) )
            {
                http_response_code( 403 );
            }
            else
            if ( is_null( $ret ) )
            {
                http_response_code( 501 );
            }
            else
            {
                http_response_code( 400 );
            }
        }
    }

    if ( string_contains( $response["error"], "DUPLICATE_REQUEST" ) )
    {
        $response["status"] = "DUPLICATE";
    }

    if ( !array_key_exists( 'accept', $request ) ) $request['accept'] = "";

    if ( is_null( $ret ) )
    {
        $response['error'] = "Could not resolve stored procedure.";
    }

    if ( is_a( $ret, 'ErrorResults' ) )
    {
        $response['severity'] = SeverityFor( $ret->sqlstate );
    }

    if ( "TRUE" == getenv( "LOG_ERRORS" ) )
    {
        if ( is_a( $ret, 'ErrorResults' ) )
        {
            switch( $ret->sqlstate )
            {
            case '45000':
                NeoLogInfo   ( "Info: "    . $ret->errorMessage );
                break;

            case '45001':
                NeoLogNotice ( "Notice: "  . $ret->errorMessage );
                break;

            case '45002':
                NeoLogWarning( "Warning: " . $ret->errorMessage );
                break;

            case '45003':
                NeoLogError  ( "Error: "   . $ret->errorMessage );
                break;
            }
        }
        else
        if ( $response['error'] )
        {
            NeoLogError( "Error: " . $response['error'] );
        }
    }

    return (object) $response;
}

function SeverityFor( $sqlstate )
{
    switch( $sqlstate )
    {
    case '45000':
        return "Info";
        break;

    case '45001':
        return "Notice";
        break;

    case '45002':
        return "WARNING";
        break;

    case '45003':
        return "ERROR";
        break;
    
    default:
        return "";
    }
}

function OutputResponse( $sp_name, $response, $request, $use_buffered = false )
{
    NeoLog( "Output Response" );
    NeoLogIn();

    $accept = DetermineAccept();

    switch( $accept )
    {
    case "application/json":
        TranslateToJSONUnbuffered( $response );
        break;

    case "text/plain":
        TranslateToPlainText( $response, $request );
        break;

    case "text/csv":
        TranslateToCSV( $response, $request );
        break;

    default:
        if ( string_has_suffix( $sp_name, "_csv" ) )
        {
            TranslateToCSV ( $response, $request );
        }
        else
        if ( $use_buffered )
        {
            TranslateToJSON( $response );
        }
        else
        {
            TranslateToJSONUnbuffered( $response );
        }
    }

    NeoLogOut();
}

function TranslateToJSON( $response )
{
    NeoLog( "Translate to JSON (Buffered)" );

    if ( ob_get_contents() ) ob_end_clean();

    set_time_limit(0);
    ignore_user_abort( TRUE );

    ob_start();
    {
        if ( getenv( "ONLY_OUTPUT_RESULT_OBJECT" ) )
        {
            if ( "ERROR" == $response->status )
            {
                $obj = json_decode( $response->error );

                if ( is_object( $obj ) )
                {
                    \JStream::EncodeObject( 0, $obj );
                }
                else
                {
                    echo $response->error;
                }
            }
            else
            if ( is_array( $response->results ) )
            {
                \JStream::EncodeArray( 0, $response->results );
            }
            else
            {
                \JStream::EncodeResultArray( 0, $response->results, "TRUE" == getenv( "NO_ARRAY_FOR_SINGLE_RESULT" ) );
            }
        }
        else
        {
            \JStream::Encode( 0, $response );
        }
        echo "\n";
        $size = ob_get_length();

        header( "Content-Length: $size"          );
        header( "Connection: close"              );
        header( "Content-Encoding: none"         );
        header( "Content-Type: application/json" );

    }
    ob_end_flush();
    ob_flush();
    flush();

    if (session_id()) session_write_close();
}

function TranslateToJSONUnbuffered( $response )
{
    NeoLog( "Translate to JSON (Unbuffered)" );

    set_time_limit(0);
    ignore_user_abort( TRUE );

    {
        header( "Connection: close"              );
        header( "Content-Encoding: none"         );
        header( "Content-Type: application/json" );

        if ( getenv( "ONLY_OUTPUT_RESULT_OBJECT" ) )
        {
            if ( "ERROR" == $response->status )
            {
                $obj = json_decode( $response->error );

                if ( is_object( $obj ) )
                {
                    \JStream::EncodeObject( 0, $obj );
                }
                else
                {
                    echo $response->error;
                }
            }
            else
            if ( is_array( $response->results ) )
            {
                \JStream::EncodeArray( 0, $response->results );
            }
            else
            {
                \JStream::EncodeResultArray( 0, $response->results, "TRUE" == getenv( "NO_ARRAY_FOR_SINGLE_RESULT" ) );
            }
        }
        else
        {
            \JStream::Encode( 0, $response );
        }
        echo "\n";
    }
    flush();

    if (session_id()) session_write_close();
}

function TranslateToCSV( $response, $request )
{
    NeoLog( "Translate to CSV" );

    if ( "OK" == $response->status )
    {
        $content  = "";
        $ds       = ("true" == array_get( $request, "double_space" ));
        $numbered = ("true" == array_get( $request, "numbered"     ));
        $map      = GenerateMap( array_get( $request, "csv_map" ) );
        $results  = $response->results;

        header( "Content-Type: application/octet-stream"                    );
        header( "Content-Disposition: attachment;filename=\"download.csv\"" );
        header( "Pragma: public"                                            );
        header( "Cache-Control: must-revalidate, post-check=0, pre-check=0" );
        header( "Connection: close" );

        if ( $map )
        {
            echo \CSV::encode( \CSV::MapValues( $map, $results->toArray() ), $numbered, $ds );
        }
        else
        {
            \CSV::Echo( $results, $numbered, $ds );
        }

        flush();
    }
}

function TranslateToPlainText( $response, $request )
{
    NeoLog( "Translate to Plain Text" );

    if ( "OK" == $response->status )
    {
        header( "Content-Type: text/plain" );

        $results = $response->results;
        $it      = $results->iterator();
        $first   = $it->hasNext() ? $it->next() : null;

        if ( $first )
        {
            foreach( $first as $key => $value )
            {
                if ( 'text' == $key )
                {
                    echo $value;
                    break;
                }
            }
        }
        flush();
    }
}

function TranslateToCSVOrig( $response, $request )
{
    NeoLog( "Translate to CSV" );

    NeoLog( var_export( $response, true ) );

    if ( "OK" == $response->status )
    {
        $content  = "";
        $ds       = ("true" == array_get( $request, "double_space" ));
        $numbered = ("true" == array_get( $request, "numbered"     ));
        $map      = GenerateMap( array_get( $request, "csv_map" ) );
        $results  = $response->results->toArray();

        if ( $map )
        {
            NeoLog( "Calling CSV::encode with mapped tuples." );

            $content = \CSV::encode( \CSV::MapValues( $map, $results ), $numbered, $ds );
        }
        else
        {
            NeoLog( "Calling CSV::encode with unmapped tuples." );

            $content = \CSV::encode( $results, $numbered, $ds );
        }

        $size = strlen( $content );

        if ( false )
        {
            error_log( "Content-Type: application/octet-stream" );
            error_log( "Content-Length: $size" );
            error_log( "Content-Disposition: attachment;filename=\"download\"" );
        }

        header( "Content-Type: application/octet-stream"                    );
        header( "Content-Length: $size"                                     );
        header( "Content-Disposition: attachment;filename=\"download.csv\"" );
        header( "Pragma: public"                                            );
        header( "Cache-Control: must-revalidate, post-check=0, pre-check=0" );
        header( "Connection: close" );
        echo $content;

        flush();
        exit;
    }
}

function GenerateMap( $base64_encoded_csv_map )
{
    $csv_map =  base64_decode( $base64_encoded_csv_map );

    if ( defined( "VERBOSE" ) && VERBOSE ) error_log( "GenerateMap( \"$csv_map\" )" );

    //
    //  csv_map=Family+Name|family_name,...
    //

    $map = null;

    if ( $csv_map )
    {
        $map  = array();
        $bits = explode( ",", $csv_map );
        
        foreach ( $bits as $mapping )
        {
            $pieces = explode( "|", $mapping );

            if ( 2 == count( $pieces ) )
            {
                $key = urldecode( $pieces[0] );
                $val = urldecode( $pieces[1] );
            
                $map[$key] = $val;
            }

            if ( defined( "VERBOSE" ) && VERBOSE ) error_log( "Mapping( $mapping ) : $key | $val" );
        }
    }
    return $map;
}

function DetermineAccept()
{
    $accept = "";

    foreach( apache_request_headers() as $header => $value )
    {
        $_value = Input::Filter( $value );

        if ( "Accept" == $header )
        {
            $accept = $value;
            break;
        }
    }

    return $accept;
}

