#!/bin/bash
#
#   For each Apache site configuration file,
#   if there is not a corresponding lets encrypt configuration file,
#   and if the site configuration file contains a SiteName directive,
#   call certbot on the ServerName specified.
#
#   If any new lets encrypt configuration files have been created,
#   and if apachectl checkconfig passes,
#   call 'service apache2 restart'
#

function usage()
{
    echo "Usage: run_certbot.sh [Apache config dir]"
    exit -1
}


function main()
{
    local apache_config_dir=$1

    echo `date`

    if [ -z "$apache_config_dir" ]
    then

        apache_config_dir="/etc/apache2"

    fi

    if [ ! -d "$apache_config_dir" ]
    then

        usage

    else

        echo "+Running certbot for Apache config dir: $apache_config_dir"

        for config_file in $( ls "$apache_config_dir/sites-enabled" ); do

            process_config_file "$apache_config_dir/sites-enabled/$config_file"

        done

    fi
}

function process_config_file()
{
    local config_file=$1
    local config_filename=`basename $config_file`
    local my_address=`dig +short myip.opendns.com @resolver1.opendns.com`

    if   string_ends_with   "$config_filename" "-le-ssl.conf"
    then
        echo "++Skipping: $config_file"

    elif string_starts_with "$config_filename" "api"
    then

        local prov_lets_encrypt_config_file="${config_file/.conf/-le-ssl.conf}"
        local servername=`grep "ServerName" "$config_file" | grep -v "#" | sed 's|\t||g' | sed 's| ||g'`
        local domain="${servername/ServerName/}"

        if [ -n "$domain" ]
        then
            if string_contains "$domain" ".local"
            then
                echo "++Ignoring local domain: $domain"

            elif [ -f "$prov_lets_encrypt_config_file" ]
            then
                echo "++OK: $domain"

            elif domain_has_tld "$domain"
            then
                echo "++Checking: $config_file, which contains domain |$domain| - looking for $prov_lets_encrypt_config_file"

                if domain_has_my_ip "$domain"
                then
                    echo "+++Running certbot for: $domain"
                    echo sudo certbot -n --agree-tos --apache --redirect --hsts --uir --expand -d "$domain" -m "webmaster@$domain"
                         sudo certbot -n --agree-tos --apache --redirect --hsts --uir --expand -d "$domain" -m "webmaster@$domain"

                    #echo sudo certbot -n --agree-tos --authenticator webroot --installer apache --webroot-path /srv/DROPSPACE/ --redirect --hsts --uir --expand -d "$domain" -d "www.$domain" -m "webmaster@$domain"
                    #     sudo certbot -n --agree-tos --authenticator webroot --installer apache --webroot-path /srv/DROPSPACE/ --redirect --hsts --uir --expand -d "$domain" -d "www.$domain" -m "webmaster@$domain"

                else
                    echo "+++Waiting for DNS update to $my_address for: $domain"

                fi

            else
                echo "++Not running certbot for non .com or .org domain: $domain"

            fi
        fi
    else

        echo "++Skipping: $config_file"

    fi
}

function domain_has_tld()
{
    local domain=$1

    if   string_ends_with "$domain" ".com"
    then
        return 0

    elif string_ends_with "$domain" ".com.au"
    then
        return 0

    elif string_ends_with "$domain" ".dev"
    then
        return 0

    elif string_ends_with "$domain" ".info"
    then
        return 0

    elif string_ends_with "$domain" ".net"
    then
        return 0

    elif string_ends_with "$domain" ".news"
    then
        return 0

    elif string_ends_with "$domain" ".org"
    then
        return 0

    elif string_ends_with "$domain" ".online"
    then
        return 0

    elif string_ends_with "$domain" ".xyz"
    then
        return 0

    else

        return -1
    fi
}

function domain_has_my_ip()
{
    local domain=$1
    local  ip_address=`dig  +noall +answer +short "$domain"`

    local  my_address=`dig +short myip.opendns.com @resolver1.opendns.com`

    if [ -z "$ip_address" ]
    then
        echo "+++Invalid IP address for $domain (should be $my_address)"
        return -1






    elif [ -z "$my_address" ]
    then
        echo "+++Invalid IP address for localhost"
        return -1

    elif string_contains "$ip_address" "$my_address"
    then
        return 0





    else
        echo "+++Invalid IP address for $domain ($ip_address) - (should be $my_address)"
        return -1

    fi
}

function string_contains()
{
    local haystack=$1
    local needle=$2

    if [ -z "${haystack##*$needle*}" ]
    then
        return 0  # success
    else
        return -1 # failure
    fi
}

function string_starts_with()
{
    local haystack=$1
    local prefix=$2

    if [ "${haystack}" = "$prefix${haystack/$prefix}" ]
    then
        return 0  # success
    else
        return -1 # failure
    fi
}

function string_ends_with()
{
    local haystack=$1
    local suffix=$2

    if [ "${haystack}" = "${haystack/$suffix}${suffix}" ]
    then
        return 0  # success
    else
        return -1 # failure
    fi
}

main "$@"
